const express = require('express');
const session = require('express-session');
const bodyParser = require('body-parser');
const path = require('path');
const sqlite3 = require('sqlite3').verbose();
const bcrypt = require('bcrypt');
const { v4: uuidv4 } = require('uuid');

const app = express();
const PORT = process.env.PORT || 3000;

app.set('view engine', 'ejs');
app.set('views', path.join(__dirname, 'views'));

app.use(express.static(path.join(__dirname, 'public')));
app.use(bodyParser.urlencoded({ extended: false }));
app.use(bodyParser.json());
app.use(session({
  secret: 'oauth-provider-secret',
  resave: false,
  saveUninitialized: false
}));

const db = new sqlite3.Database(':memory:');

db.serialize(() => {
  db.run(`
    CREATE TABLE IF NOT EXISTS users (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      username TEXT UNIQUE NOT NULL,
      email TEXT UNIQUE NOT NULL,
      password TEXT NOT NULL,
      name TEXT,
      profile_picture TEXT,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS oauth_clients (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      client_id TEXT UNIQUE NOT NULL,
      client_secret TEXT NOT NULL,
      redirect_uri TEXT NOT NULL,
      name TEXT NOT NULL,
      user_id INTEGER,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (user_id) REFERENCES users (id)
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS oauth_codes (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      code TEXT UNIQUE NOT NULL,
      client_id TEXT NOT NULL,
      user_id INTEGER NOT NULL,
      expires_at DATETIME NOT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (user_id) REFERENCES users (id)
    )
  `);

  db.run(`
    CREATE TABLE IF NOT EXISTS oauth_tokens (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      access_token TEXT UNIQUE NOT NULL,
      refresh_token TEXT UNIQUE,
      client_id TEXT NOT NULL,
      user_id INTEGER NOT NULL,
      expires_at DATETIME NOT NULL,
      created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
      FOREIGN KEY (user_id) REFERENCES users (id)
    )
  `);

  const clientId = 'test-client-id';
  const clientSecret = 'test-client-secret';
  
  db.get('SELECT * FROM oauth_clients WHERE client_id = ?', [clientId], (err, client) => {
    if (err) {
      console.error('Error checking for existing client:', err);
      return;
    }
    
    if (client) {
      db.run(`
        UPDATE oauth_clients 
        SET redirect_uri = ? 
        WHERE client_id = ?
      `, ['http://localhost:3001/auth/callback http://localhost:3001/auth/callback-modal', clientId]);
    } else {
      db.run(`
        INSERT INTO oauth_clients (client_id, client_secret, redirect_uri, name, user_id)
        VALUES (?, ?, ?, ?, ?)
      `, [clientId, clientSecret, 'http://localhost:3001/auth/callback http://localhost:3001/auth/callback-modal', 'Test Client', null]);
    }
  });
});

app.locals.db = db;

app.get('/', (req, res) => {
  res.render('index', { user: req.session.user });
});

const authRoutes = require('./routes/auth');
const userRoutes = require('./routes/user');
const oauthRoutes = require('./routes/oauth');

app.use('/auth', authRoutes);
app.use('/user', userRoutes);
app.use('/oauth', oauthRoutes);

app.listen(PORT, () => {
  console.log(`OAuth Provider running on http://localhost:${PORT}`);
});

module.exports = { app };
