"use strict";

const fetch = require("node-fetch");
const { MoleculerError } = require("moleculer").Errors;

module.exports = {
	name: "uptime",

	settings: {
		serviceEndpoints: {
			stocks: "http://localhost:3000/api/stocks/status",
			orders: "http://localhost:3000/api/orders/status",
			invoices: "http://localhost:3000/api/invoices/status",
			pdf: "http://localhost:3000/api/pdf/status",
		},
	},

	actions: {
		status: {
			rest: "GET /status",
			async handler() {
				return {
					status: "ok",
					service: "uptime",
					timestamp: new Date().toISOString(),
				};
			},
		},

		check: {
			rest: "GET /check",
			params: {
				service: { type: "string" },
			},
			async handler(ctx) {
				const { service } = ctx.params;
				let url;

				if (service.startsWith("http://") || service.startsWith("https://")) {
					url = service;
				} else {
					url = this.settings.serviceEndpoints[service];
					if (!url) {
						throw new MoleculerError(`Unknown service: ${service}`, 404, "SERVICE_NOT_FOUND");
					}
				}

				try {
					const response = await fetch(url);
					return {
						service: service,
						url: url,
						status: response.status === 200 ? "up" : "down",
						statusCode: response.status,
						timestamp: new Date().toISOString(),
					};
				} catch (error) {
					return {
						service: service,
						url: url,
						status: "down",
						error: error.message,
						timestamp: new Date().toISOString(),
					};
				}
			},
		},
	},
};
