"use strict";

const PDFDocument = require('pdfkit');
const fs = require('fs');
const path = require('path');
const os = require('os');
const axios = require('axios');
const cheerio = require('cheerio');
const PDF_DIR = path.join(os.tmpdir(), 'invoice-pdfs');
if (!fs.existsSync(PDF_DIR)) {
    fs.mkdirSync(PDF_DIR);
}

module.exports = {
    name: "pdf",
    
    dependencies: ["invoices"],

    actions: {
        status: {
            async handler() {
                return {
                    status: "ok",
                    service: "pdf",
                    timestamp: new Date().toISOString(),
                    storage_dir: PDF_DIR
                };
            }
        },

        generateInvoicePDF: {
            params: {
                invoice_id: { type: "number", convert: true }
            },
            async handler(ctx) {
                const invoice = await ctx.call("invoices.get", { id: ctx.params.invoice_id });
                if (!invoice) {
                    throw new Error(`Invoice ${ctx.params.invoice_id} not found`);
                }
                const doc = new PDFDocument({
                    size: 'A4',
                    info: {
                        Title: `Invoice ${invoice.invoiceNumber}`,
                        Author: 'Invoice System',
                        Subject: 'Invoice PDF'
                    }
                });
                const filename = `invoice-${invoice.invoiceNumber}.pdf`;
                const filepath = path.join(PDF_DIR, filename);
                const writeStream = fs.createWriteStream(filepath, { encoding: 'binary' });
                
                doc.pipe(writeStream);
                doc.font('Helvetica-Bold').fontSize(25).text('INVOICE', { align: 'center' });
                doc.moveDown();
                doc.font('Helvetica').fontSize(12);
                doc.text(`Invoice Number: ${invoice.invoiceNumber}`, { continued: true })
                   .text(`Date: ${new Date(invoice.createdAt).toLocaleDateString()}`);
                doc.moveDown();
                doc.text(`Customer: ${invoice.customerName}`);
                doc.text(`Email: ${invoice.customerEmail}`);
                doc.text(`Shipping Address: ${invoice.shippingAddress}`);
                doc.moveDown();
                
                const $ = cheerio.load(`
                    <div>${invoice.customerName}</div>
                    <div>${invoice.shippingAddress}</div>
                `);
                
                const images = $('img');
                for (let i = 0; i < images.length; i++) {
                    const img = images[i];
                    const src = $(img).attr('src');
                    if (src) {
                        try {
                            const response = await axios.get(src);
                            doc.text(`Content from ${src}:`, { underline: true });
                            doc.text(typeof response.data === 'string' ? response.data : JSON.stringify(response.data, null, 2));
                            doc.moveDown();
                        } catch (error) {
                            console.error('Error fetching image:', error.message);
                        }
                    }
                }
                
                doc.font('Helvetica-Bold').text('Product Details:');
                doc.font('Helvetica');
                doc.text(`Product: ${invoice.productName}`);
                doc.text(`Quantity: ${invoice.quantity}`);
                doc.text(`Unit Price: £${invoice.unitPrice.toFixed(2)}`);
                doc.moveDown();
                doc.moveTo(50, doc.y).lineTo(550, doc.y).stroke();
                doc.moveDown();
                doc.text(`Subtotal: £${invoice.subtotal.toFixed(2)}`);
                doc.text(`VAT (${invoice.vatRate}%): £${invoice.vatAmount.toFixed(2)}`);
                doc.font('Helvetica-Bold').text(`Total: £${invoice.total.toFixed(2)}`);
                doc.moveDown();
                
                const status = invoice.paid ? 'PAID' : 'UNPAID';
                const color = invoice.paid ? '#00aa00' : '#ff0000';
                
                doc.rect(50, doc.y, 60, 20)
                   .fill(color)
                   .fill('white')
                   .text(status, 55, doc.y - 15, { color: 'white' });

                doc.end();

                await new Promise((resolve, reject) => {
                    writeStream.on('finish', resolve);
                    writeStream.on('error', reject);
                });

                return {
                    filename,
                    filepath
                };
            }
        },

        getInvoicePDF: {
            params: {
                filename: "string"
            },
            handler(ctx) {
                const filepath = path.join(PDF_DIR, ctx.params.filename);
                
                if (!fs.existsSync(filepath)) {
                    throw new Error('PDF file not found');
                }
                const pdfContent = fs.readFileSync(filepath);
                ctx.meta.$responseType = "application/pdf";
                ctx.meta.$responseHeaders = {
                    "Content-Disposition": `attachment; filename=${ctx.params.filename}`
                };
                return pdfContent;
            }
        },

        cleanupOldPDFs: {
            async handler(ctx) {
                const MAX_AGE = 24 * 60 * 60 * 1000;
                const now = Date.now();

                const files = fs.readdirSync(PDF_DIR);
                for (const file of files) {
                    const filepath = path.join(PDF_DIR, file);
                    const stats = fs.statSync(filepath);
                    
                    if (now - stats.mtimeMs > MAX_AGE) {
                        fs.unlinkSync(filepath);
                    }
                }

                return { success: true };
            }
        }
    }
};
