"use strict";

const VAT_RATES = {
    UK: 0.20,
    EU: 0.21,
    US: 0.00,
    DEFAULT: 0.20
};

module.exports = {
    name: "invoices",
    
    dependencies: ["database"],

    actions: {
        status: {
            async handler() {
                return {
                    status: "ok",
                    service: "invoices",
                    timestamp: new Date().toISOString()
                };
            }
        },

        generate: {
            params: {
                order_id: { type: "number", convert: true },
                total_amount: { type: "number", convert: true },
                customer_name: "string",
                customer_email: "string",
                region: { type: "string", optional: true, default: "UK" }
            },
            async handler(ctx) {
                const { order_id, total_amount, customer_name, customer_email, region } = ctx.params;
                const vatRate = VAT_RATES[region] || VAT_RATES.DEFAULT;
                const subtotal = total_amount / (1 + vatRate);
                const vatAmount = total_amount - subtotal;
                const date = new Date();
                const year = date.getFullYear();
                const month = String(date.getMonth() + 1).padStart(2, '0');
                const lastInvoice = await ctx.call("database.queryOne", {
                    sql: "SELECT invoice_number FROM invoices WHERE invoice_number LIKE ? ORDER BY id DESC LIMIT 1",
                    params: [`${year}-${month}-%`]
                });
                
                let sequence = 1;
                if (lastInvoice) {
                    sequence = parseInt(lastInvoice.invoice_number.split('-')[2]) + 1;
                }
                
                const invoiceNumber = `${year}-${month}-${String(sequence).padStart(5, '0')}`;
                const result = await ctx.call("database.execute", {
                    sql: `
                        INSERT INTO invoices (
                            order_id, invoice_number, customer_name, customer_email,
                            subtotal, vat_rate, vat_amount, total_amount, region
                        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
                    `,
                    params: [
                        order_id,
                        invoiceNumber,
                        customer_name,
                        customer_email,
                        subtotal,
                        vatRate,
                        vatAmount,
                        total_amount,
                        region
                    ]
                });

                return this.actions.get({ id: result.lastInsertRowid });
            }
        },

        get: {
            params: {
                id: { type: "number", convert: true }
            },
            async handler(ctx) {
                const invoice = await ctx.call("database.queryOne", {
                    sql: `
                        SELECT 
                            i.*,
                            o.shipping_address,
                            oi.quantity,
                            oi.price_at_time as unit_price,
                            s.name as product_name
                        FROM invoices i
                        JOIN orders o ON i.order_id = o.id
                        JOIN order_items oi ON o.id = oi.order_id
                        JOIN stocks s ON oi.stock_id = s.id
                        WHERE i.id = ?
                    `,
                    params: [ctx.params.id]
                });
                
                if (!invoice) return null;
                const subtotal = invoice.quantity * invoice.unit_price;
                return {
                    ...invoice,
                    total: invoice.total_amount,
                    paid: invoice.status === 'paid',
                    orderId: invoice.order_id,
                    customerName: invoice.customer_name,
                    customerEmail: invoice.customer_email,
                    createdAt: invoice.created_at,
                    invoiceNumber: invoice.invoice_number,
                    shippingAddress: invoice.shipping_address,
                    productName: invoice.product_name,
                    quantity: invoice.quantity,
                    unitPrice: invoice.unit_price,
                    subtotal: subtotal,
                    vatRate: invoice.vat_rate * 100,
                    vatAmount: invoice.vat_amount,
                    total: invoice.total_amount
                };
            }
        },

        getByOrder: {
            params: {
                order_id: { type: "number", convert: true }
            },
            async handler(ctx) {
                const invoice = await ctx.call("database.queryOne", {
                    sql: 'SELECT * FROM invoices WHERE order_id = ?',
                    params: [ctx.params.order_id]
                });
                
                if (!invoice) return null;
                return {
                    ...invoice,
                    total: invoice.total_amount,
                    paid: invoice.status === 'paid',
                    orderId: invoice.order_id,
                    customerName: invoice.customer_name,
                    customerEmail: invoice.customer_email,
                    createdAt: invoice.created_at,
                    invoiceNumber: invoice.invoice_number
                };
            }
        },

        list: {
            async handler(ctx) {
                const invoices = await ctx.call("database.query", {
                    sql: 'SELECT * FROM invoices ORDER BY created_at DESC'
                });
                return invoices.map(invoice => ({
                    ...invoice,
                    total: invoice.total_amount,
                    paid: invoice.status === 'paid',
                    orderId: invoice.order_id,
                    customerName: invoice.customer_name,
                    customerEmail: invoice.customer_email,
                    createdAt: invoice.created_at,
                    invoiceNumber: invoice.invoice_number
                }));
            }
        },

        markAsPaid: {
            params: {
                id: { type: "number", convert: true }
            },
            async handler(ctx) {
                const invoice = await ctx.call("database.queryOne", {
                    sql: 'SELECT * FROM invoices WHERE id = ?',
                    params: [ctx.params.id]
                });

                if (!invoice) {
                    throw new Error(`Invoice with ID ${ctx.params.id} not found`);
                }

                if (invoice.status === 'paid') {
                    throw new Error(`Invoice ${ctx.params.id} is already paid`);
                }
                await ctx.call("database.execute", {
                    sql: 'UPDATE invoices SET status = ?, paid_at = CURRENT_TIMESTAMP WHERE id = ?',
                    params: ['paid', ctx.params.id]
                });

                return this.actions.get({ id: ctx.params.id });
            }
        },

        calculateVAT: {
            params: {
                amount: "number",
                region: { type: "string", optional: true, default: "UK" }
            },
            handler(ctx) {
                const { amount, region } = ctx.params;
                const vatRate = VAT_RATES[region] || VAT_RATES.DEFAULT;
                const subtotal = amount / (1 + vatRate);
                const vatAmount = amount - subtotal;

                return {
                    subtotal: Number(subtotal.toFixed(2)),
                    vatRate,
                    vatAmount: Number(vatAmount.toFixed(2)),
                    total: amount
                };
            }
        }
    }
};
