"use strict";

const Database = require("better-sqlite3");
const path = require("path");
const db = new Database("store.db");

db.exec(`
    DROP TABLE IF EXISTS invoices;
    DROP TABLE IF EXISTS order_items;
    DROP TABLE IF EXISTS orders;
    DROP TABLE IF EXISTS stocks;
`);

db.exec(`
    CREATE TABLE IF NOT EXISTS stocks (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        name TEXT NOT NULL,
        quantity INTEGER NOT NULL,
        price REAL NOT NULL
    );

    CREATE TABLE IF NOT EXISTS orders (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        customer_name TEXT NOT NULL,
        customer_email TEXT NOT NULL,
        shipping_address TEXT NOT NULL,
        status TEXT NOT NULL DEFAULT 'pending',
        total_amount REAL NOT NULL,
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
    );

    CREATE TABLE IF NOT EXISTS order_items (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        order_id INTEGER NOT NULL,
        stock_id INTEGER NOT NULL,
        quantity INTEGER NOT NULL,
        price_at_time REAL NOT NULL,
        FOREIGN KEY (order_id) REFERENCES orders(id),
        FOREIGN KEY (stock_id) REFERENCES stocks(id)
    );

    CREATE TABLE IF NOT EXISTS invoices (
        id INTEGER PRIMARY KEY AUTOINCREMENT,
        order_id INTEGER NOT NULL,
        invoice_number TEXT NOT NULL,
        customer_name TEXT NOT NULL,
        customer_email TEXT NOT NULL,
        subtotal REAL NOT NULL,
        vat_rate REAL NOT NULL,
        vat_amount REAL NOT NULL,
        total_amount REAL NOT NULL,
        region TEXT NOT NULL DEFAULT 'UK',
        status TEXT NOT NULL DEFAULT 'issued',
        created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
        paid_at DATETIME,
        FOREIGN KEY (order_id) REFERENCES orders(id)
    );
`);

const initializeProducts = () => {
    const products = [
        { name: "Premium Copy Paper (500 sheets)", quantity: 1000, price: 6.99 },
        { name: "Dunder Mifflin Basic Paper Ream", quantity: 800, price: 4.99 },
        { name: "Colored Paper Assortment", quantity: 500, price: 8.99 },
        { name: "Recycled Paper Bundle", quantity: 600, price: 5.99 },
        { name: "Watermark Paper Premium", quantity: 300, price: 12.99 },
        { name: "Card Stock Heavy Duty", quantity: 400, price: 15.99 },
        { name: "Photo Paper Glossy", quantity: 200, price: 19.99 },
        { name: "Legal Size Paper Bundle", quantity: 450, price: 9.99 }
    ];

    const stmt = db.prepare("INSERT INTO stocks (name, quantity, price) VALUES (?, ?, ?)");
    products.forEach(product => {
        stmt.run(product.name, product.quantity, product.price);
    });
};

const initializeOrders = () => {
    const getStockId = (name) => {
        const stock = db.prepare("SELECT id FROM stocks WHERE name = ?").get(name);
        return stock.id;
    };

    const orders = [
        {
            customer_name: "Jeremy",
            customer_email: "jeremy@jeremy.com",
            shipping_address: "123 Jeremy Lane",
            items: [
                { product: "Premium Copy Paper (500 sheets)", quantity: 50 }
            ]
        },
        {
            customer_name: "Jeremy",
            customer_email: "jeremy@jeremy.com",
            shipping_address: "123 Jeremy Lane",
            items: [
                { product: "Watermark Paper Premium", quantity: 5 }
            ]
        },
        {
            customer_name: "Jeremy",
            customer_email: "jeremy@jeremy.com",
            shipping_address: "123 Jeremy Lane",
            items: [
                { product: "Photo Paper Glossy", quantity: 50 }
            ]
        }
    ];

    orders.forEach(order => {
        let total_amount = 0;
        const orderItems = [];

        order.items.forEach(item => {
            const stock = db.prepare("SELECT id, price FROM stocks WHERE name = ?").get(item.product);
            total_amount += stock.price * item.quantity;
            orderItems.push({
                stock_id: stock.id,
                quantity: item.quantity,
                price_at_time: stock.price
            });
        });

        const orderResult = db.prepare(
            'INSERT INTO orders (customer_name, customer_email, shipping_address, total_amount) VALUES (?, ?, ?, ?)'
        ).run(order.customer_name, order.customer_email, order.shipping_address, total_amount);

        const orderId = orderResult.lastInsertRowid;

        const itemStmt = db.prepare(
            'INSERT INTO order_items (order_id, stock_id, quantity, price_at_time) VALUES (?, ?, ?, ?)'
        );

        orderItems.forEach(item => {
            itemStmt.run(orderId, item.stock_id, item.quantity, item.price_at_time);
        });

        const vatRate = 0.20; // 20% VAT
        const subtotal = total_amount / (1 + vatRate);
        const vatAmount = total_amount - subtotal;
        const date = new Date();
        const year = date.getFullYear();
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const lastInvoice = db.prepare(
            "SELECT invoice_number FROM invoices WHERE invoice_number LIKE ? ORDER BY id DESC LIMIT 1"
        ).get(`${year}-${month}-%`);
        
        let sequence = 1;
        if (lastInvoice) {
            sequence = parseInt(lastInvoice.invoice_number.split('-')[2]) + 1;
        }
        
        const invoiceNumber = `${year}-${month}-${String(sequence).padStart(5, '0')}`;

        db.prepare(`
            INSERT INTO invoices (
                order_id, invoice_number, customer_name, customer_email,
                subtotal, vat_rate, vat_amount, total_amount, region
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)
        `).run(
            orderId,
            invoiceNumber,
            order.customer_name,
            order.customer_email,
            subtotal,
            vatRate,
            vatAmount,
            total_amount,
            'UK'
        );
    });
};

module.exports = {
    name: "database",

    actions: {
        query: {
            params: {
                sql: "string",
                params: { type: "array", optional: true }
            },
            handler(ctx) {
                const stmt = db.prepare(ctx.params.sql);
                return stmt.all(ctx.params.params || []);
            }
        },

        queryOne: {
            params: {
                sql: "string",
                params: { type: "array", optional: true }
            },
            handler(ctx) {
                const stmt = db.prepare(ctx.params.sql);
                return stmt.get(ctx.params.params || []);
            }
        },

        execute: {
            params: {
                sql: "string",
                params: { type: "array", optional: true }
            },
            handler(ctx) {
                const stmt = db.prepare(ctx.params.sql);
                return stmt.run(ctx.params.params || []);
            }
        },

        transaction: {
            handler(ctx) {
                return db.transaction();
            }
        }
    },

    created() {
        this.db = db;
        initializeProducts();
        initializeOrders();
    }
};
