"use strict";

const ApiGateway = require("moleculer-web");
const path = require("path");
const fs = require('fs');

module.exports = {
    name: "api",
    mixins: [ApiGateway],

    actions: {
        serveIndex(ctx) {
            return fs.readFileSync(path.resolve(__dirname, "../test.html"), "utf8");
        }
    },

    settings: {
        port: 3000,
        cors: {
            origin: "*",
            methods: ["GET", "OPTIONS", "POST", "PUT", "DELETE"],
            allowedHeaders: ["Content-Type"]
        },
        routes: [
            {
                path: "/",
                whitelist: ["api.*"],
                cors: true,
                aliases: {
                    "GET /": "api.serveIndex"
                },
                onBeforeCall(ctx, route, req, res) {
                    ctx.meta.$responseType = "text/html";
                },
                callOptions: {
                    meta: {
                        $responseType: "text/html"
                    }
                }
            },
            {
                path: "/api",
            whitelist: [
                "stocks.*",
                "orders.*",
                "invoices.*",
                "pdf.*",
                "uptime.*"
            ],
            cors: true,
            bodyParser: true,
            autoConvert: true,

            aliases: {
                "GET /stocks/status": "stocks.status",
                "GET /stocks": "stocks.list",
                "GET /stocks/:id": {
                    action: "stocks.get",
                    params: {
                        id: { type: "number", convert: true }
                    }
                },
                "POST /stocks": {
                    action: "stocks.create",
                    onBeforeCall(ctx, route, req, res) {
                        console.log("Creating stock - Raw request:", {
                            body: req.body,
                            headers: req.headers
                        });
                    },
                    onError(ctx, route, req, res, err) {
                        console.error("Stock creation error:", {
                            message: err.message,
                            type: err.type,
                            code: err.code,
                            data: err.data
                        });
                        throw err;
                    }
                },
                "PUT /stocks/:id": {
                    action: "stocks.update",
                    params: {
                        id: { type: "number", convert: true }
                    }
                },
                "PUT /stocks/:id/quantity": {
                    action: "stocks.updateQuantity",
                    params: {
                        id: { type: "number", convert: true }
                    }
                },

                "GET /orders/status": "orders.status",
                "GET /orders": "orders.list",
                "GET /orders/:id": {
                    action: "orders.get",
                    params: {
                        id: { type: "number", convert: true }
                    }
                },
                "POST /orders": "orders.create",
                "PUT /orders/:id/status": {
                    action: "orders.updateStatus",
                    params: {
                        id: { type: "number", convert: true },
                        status: { type: "string", enum: ["pending", "processing", "completed", "cancelled"] }
                    },
                    onBeforeCall(ctx, route, req, res) {
                        ctx.broker.logger.debug("Order status update - Raw parameters:", {
                            params: req.params,
                            body: req.body,
                            query: req.query
                        });

                        const id = parseInt(req.params.id, 10);
                        if (!id || isNaN(id)) {
                            throw new Error(`Invalid order ID: ${req.params.id}`);
                        }

                        const status = req.body.status || req.query.status;
                        if (!status) {
                            throw new Error("Missing required parameter: status");
                        }

                        ctx.params = { id, status };
                        ctx.broker.logger.debug("Order status update - Processed parameters:", ctx.params);
                        
                        return ctx.params;
                    }
                },

                "GET /invoices/status": "invoices.status",
                "GET /invoices": "invoices.list",
                "GET /invoices/:id": {
                    action: "invoices.get",
                    params: {
                        id: { type: "number", convert: true }
                    }
                },
                "GET /invoices/order/:orderId": {
                    action: "invoices.getByOrder",
                    params: {
                        orderId: { type: "number", convert: true }
                    }
                },
                "PUT /invoices/:id/pay": {
                    action: "invoices.markAsPaid",
                    params: {
                        id: { type: "number", convert: true }
                    },
                    onBeforeCall(ctx, route, req, res) {
                        ctx.broker.logger.debug("Invoice pay - Raw parameters:", {
                            params: req.params,
                            body: req.body,
                            query: req.query
                        });

                        const id = parseInt(req.params.id, 10);
                        if (!id || isNaN(id)) {
                            throw new Error(`Invalid invoice ID: ${req.params.id}`);
                        }

                        ctx.params = { id };
                        ctx.broker.logger.debug("Invoice pay - Processed parameters:", ctx.params);
                        
                        return ctx.params;
                    }
                },
                
                "POST /vat/calculate": "invoices.calculateVAT",
                "GET /pdf/status": "pdf.status",
                "POST /pdf/generate": {
                    action: "pdf.generateInvoicePDF",
                    params: {
                        invoice_id: { type: "number", convert: true }
                    }
                },
                "GET /pdf/get/:filename": {
                    action: "pdf.getInvoicePDF",
                    params: {
                        filename: "string"
                    }
                },

                "GET /uptime/check": {
                    action: "uptime.check",
                    params: {
                        service: { type: "string" }
                    }
                },
                "GET /uptime/status": "uptime.status"
            },

            mappingPolicy: "restrict",
            parseQueryParams: true
        }
        ]
    }
};
