# Stock Tracking Microservice

A microservices application built with Node.js, Moleculer, and SQLite for managing stock inventory, orders, and invoices.

## Project Structure

```
test-microservice/
├── services/
│   ├── api.service.js     # API Gateway service
│   ├── database.service.js # Database service
│   ├── stocks.service.js  # Stock management service
│   ├── orders.service.js  # Order management service
│   └── invoices.service.js # Invoice and VAT calculation service
├── app.js                # Main application file
├── package.json         # Project dependencies
└── store.db           # SQLite database
```

## Technology Stack

- **Node.js** - Runtime environment
- **Moleculer** - Microservices framework
- **SQLite** - Database (using better-sqlite3)
- **Moleculer Web** - API Gateway

## Getting Started

1. Install dependencies:
```bash
npm install
```

2. Start the service:
```bash
npm run start:all
```

The service will run on http://localhost:3000/api

## API Endpoints

### Stock Management

#### List All Stock Items
```bash
curl http://localhost:3000/api/stocks
```
Response:
```json
[
  {
    "id": 1,
    "name": "Test Product",
    "quantity": 100,
    "price": 9.99
  }
]
```

#### Get Specific Stock Item
```bash
curl http://localhost:3000/api/stocks/1
```

#### Create New Stock Item
```bash
curl -X POST http://localhost:3000/api/stocks \
  -H "Content-Type: application/json" \
  -d '{
    "name": "New Product",
    "quantity": 50,
    "price": 19.99
  }'
```

#### Update Stock Item
```bash
curl -X PUT http://localhost:3000/api/stocks/1 \
  -H "Content-Type: application/json" \
  -d '{
    "name": "Updated Product",
    "quantity": 75,
    "price": 24.99
  }'
```

#### Update Stock Quantity
```bash
curl -X PUT http://localhost:3000/api/stocks/1/quantity \
  -H "Content-Type: application/json" \
  -d '{
    "quantity": 60
  }'
```

#### Delete Stock Item
```bash
curl -X DELETE http://localhost:3000/api/stocks/1
```

### Order Management

#### Create New Order
```bash
curl -X POST http://localhost:3000/api/orders \
  -H "Content-Type: application/json" \
  -d '{
    "customer_name": "John Doe",
    "customer_email": "john@example.com",
    "items": [
      {
        "stock_id": 1,
        "quantity": 2
      }
    ]
  }'
```
Response:
```json
{
  "id": 1,
  "customer_name": "John Doe",
  "customer_email": "john@example.com",
  "status": "pending",
  "total_amount": 19.98,
  "created_at": "2024-12-06T19:40:00.000Z",
  "updated_at": "2024-12-06T19:40:00.000Z",
  "items": [
    {
      "id": 1,
      "order_id": 1,
      "stock_id": 1,
      "quantity": 2,
      "price_at_time": 9.99
    }
  ]
}
```

#### List All Orders
```bash
curl http://localhost:3000/api/orders
```

#### Get Specific Order
```bash
curl http://localhost:3000/api/orders/1
```

#### Update Order Status
```bash
curl -X PUT http://localhost:3000/api/orders/1/status \
  -H "Content-Type: application/json" \
  -d '{
    "status": "completed"
  }'
```

### Invoice Management

#### Get Invoice by ID
```bash
curl http://localhost:3000/api/invoices/1
```

#### Get Invoice by Order ID
```bash
curl http://localhost:3000/api/invoices/order/1
```

#### List All Invoices
```bash
curl http://localhost:3000/api/invoices
```

#### Mark Invoice as Paid
```bash
curl -X PUT http://localhost:3000/api/invoices/1/pay
```

#### Calculate VAT
```bash
curl -X POST http://localhost:3000/api/vat/calculate \
  -H "Content-Type: application/json" \
  -d '{
    "amount": 100,
    "region": "UK"
  }'
```
Response:
```json
{
  "subtotal": 83.33,
  "vatRate": 0.20,
  "vatAmount": 16.67,
  "total": 100
}
```

## Database Schema

### Store Database (store.db)

#### Stocks Table
```sql
CREATE TABLE stocks (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    name TEXT NOT NULL,
    quantity INTEGER NOT NULL,
    price REAL NOT NULL
)
```

#### Orders Table
```sql
CREATE TABLE orders (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    customer_name TEXT NOT NULL,
    customer_email TEXT NOT NULL,
    status TEXT NOT NULL DEFAULT 'pending',
    total_amount REAL NOT NULL,
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    updated_at DATETIME DEFAULT CURRENT_TIMESTAMP
)

CREATE TABLE order_items (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    order_id INTEGER NOT NULL,
    stock_id INTEGER NOT NULL,
    quantity INTEGER NOT NULL,
    price_at_time REAL NOT NULL,
    FOREIGN KEY (order_id) REFERENCES orders(id),
    FOREIGN KEY (stock_id) REFERENCES stocks(id)
)
```

#### Invoices Table
```sql
CREATE TABLE invoices (
    id INTEGER PRIMARY KEY AUTOINCREMENT,
    order_id INTEGER NOT NULL,
    invoice_number TEXT NOT NULL,
    customer_name TEXT NOT NULL,
    customer_email TEXT NOT NULL,
    subtotal REAL NOT NULL,
    vat_rate REAL NOT NULL,
    vat_amount REAL NOT NULL,
    total_amount REAL NOT NULL,
    region TEXT NOT NULL DEFAULT 'UK',
    status TEXT NOT NULL DEFAULT 'issued',
    created_at DATETIME DEFAULT CURRENT_TIMESTAMP,
    paid_at DATETIME,
    FOREIGN KEY (order_id) REFERENCES orders(id)
)
```

## VAT Rates by Region

- UK: 20%
- EU: 21%
- US: 0% (No VAT)
- Default: 20%

## Error Handling

The service includes error handling for:
- Invalid requests
- Missing parameters
- Insufficient stock
- Database errors
- Non-existent items
- Invalid order status transitions
- VAT calculation errors

Error responses will include an appropriate HTTP status code and error message.

## Development Notes

- The service automatically creates required database tables
- Orders automatically trigger invoice generation
- Stock quantities are automatically updated when orders are placed
- Invoices include VAT calculations based on region
- All endpoints are accessible through the API Gateway at /api
- The service includes input validation for all endpoints
