/* iface-wlan-14.c
 * 
 * This file contains all code sepecific to the wireless device interface. 
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <errno.h>

#include <sys/ioctl.h>
#include <sys/socket.h>
#include <linux/types.h>	/* for "__kernel_caddr_t" et al */
#include <linux/socket.h>	/* for "struct sockaddr" et al  */
#include <linux/wireless.h>

#include "pcap.h"
#include "prismstumbler.h"
#include "helper.h"
#include "iface-wlan-14.h"

#ifdef WLAN_NG
#include "wlanctl/wlanctl.h"
#warning WLAN-NG active
#endif

#ifndef DLT_PRISM_HEADER
#define DLT_PRISM_HEADER 119
#endif

extern psconfig_t cfg;
extern int RawSock;

typedef struct CaptureArg_t
{
	pcap_t *pcap;
	pcap_dumper_t *dump;
	int offset;
}
CaptureArg;

static int openCard (void);
static CaptureArg ca;
static char errbuf[PCAP_ERRBUF_SIZE];
static int lastchan = -1;

extern int g_offset;
extern int have_led;
char logbuf[120];
extern int QUIET_MODE;

/* pcap dump stuff */
int
pcap_start_dump()
{
	if (ca.pcap)
		ca.dump = pcap_dump_open(ca.pcap,cfg.dumpfile);
	if (ca.dump) return TRUE;
		else
		{
			fprintf(stderr,"%s\n",pcap_geterr(ca.pcap));
			return FALSE;
		}	
}

void
pcap_stop_dump()
{
	if (ca.dump)
		pcap_dump_close(ca.dump);
	ca.dump = NULL;
}

/* package capture interface */

/*
 * Open the Capture device 
 */
int
openPacket (void)
{
	if (openCard() == FALSE) 
		return -1;
#ifdef DEBUG
	printf ("opening dev %s\n", cfg.device);
#endif
	ca.dump = NULL;
	ca.pcap = NULL;
	
	/* destinct handling for different data gathering modes */
	if (cfg.devtype != DT_SCAN)
	{
		ca.pcap = pcap_open_live (cfg.device, MAX_BUFFER_SIZE, 1, 4000, errbuf);
		if (ca.pcap)
		{
	
			pcap_setnonblock (ca.pcap, 1, errbuf);
	
			switch (pcap_datalink (ca.pcap))
			{
			case DLT_PRISM_HEADER:
				sprintf (logbuf, "pcap: DLT_PRISM_HEADER\n");
				ca.offset = 144;
				break;
			case DLT_IEEE802_11:
				sprintf (logbuf, "pcap: DLT_IEEE802_11\n");
				ca.offset = 0;
				break;
			case DLT_AIRONET_HEADER:
				sprintf (logbuf, "pcap: DLT_AIRONET_HEADER\n");
				ca.offset = 0;
				break;
			default:
				sprintf (logbuf, "pcap: COOKED %d\n",pcap_datalink (ca.pcap));
				ca.offset = 160;
			}
	#ifdef DEBUG
			printf ("%s\n", logbuf);
	#endif
			g_offset = ca.offset;
			if (have_led)
				system("/usr/bin/led blink 5 5");
			if (cfg.dumptofile) pcap_start_dump();
			return 1;
		}
		
	#ifdef DEBUG
			printf ("pcap init interface failed\n");
	#endif
		return -1;
	}
	else /* no monitor mode, scan only */
	{
		if (have_led)
			system("/usr/bin/led blink 5 5");
		return 1;
	}
}


int
getPacket (unsigned char *buf, int maxlen, int timeout)
{
	struct pcap_pkthdr pktHdr;
	u_char *ret;

	ret = (u_char *) pcap_next (ca.pcap, &pktHdr);
	if (ret)
	{
		memcpy (buf, ret, pktHdr.len);
		if (ca.dump) 
			pcap_dump((u_char *)ca.dump,&pktHdr,ret);
		return pktHdr.len;
	}
	else
	{
		return 0;
	}
}


/*
 * Shutdown packet capture facility.
 */
void
closePacket ()
{
	if (ca.dump)
		pcap_stop_dump();
	if (ca.pcap)
		pcap_close (ca.pcap);
	if (have_led)
		system("/usr/bin/led off");
}


void
shutCard (void)
{
	int fd;
	struct iwreq ireq;	//for Orinoco
	int *ptr;
	char syscmd[30];	
#ifdef WLAN_NG
	char *argv[10];
	int argc;
#endif
#ifdef DEBUG
			fprintf (stderr,
				 "Shutting down card.\n");
#endif	
	switch (cfg.devtype)
	{
	case DT_HOSTAP:
		sprintf (syscmd, "/sbin/iwconfig %s mode managed", cfg.device);
		system (syscmd);
		break;

	case DT_ORINOCO:
	 	fd = socket (AF_INET, SOCK_STREAM, 0);
		if (fd == -1)
		{
			if (!QUIET_MODE) fprintf (stderr,
				 "Error opening socket for orinoco card: %s.\n",
				 strerror (errno));
			RawSock = -1;
			return;
		}
		ptr = (int *) ireq.u.name;
		ptr[0] = 0;
		ptr[1] = 0;
		strcpy (ireq.ifr_ifrn.ifrn_name, cfg.device);
		if (ioctl (fd, SIOCIWFIRSTPRIV + 0x8, &ireq) != 0)
		{
			if (!QUIET_MODE) fprintf (stderr,
				 "Error setting channel for orinoco card: %s.\n",
				 strerror (errno));
		}
		close (fd);
		break;
#ifdef WLAN_NG
	case DT_PRISM:
		memset (argv, 0, 10);
		sprintf (syscmd, "/sbin/ifconfig %s down", cfg.device);
		system (syscmd);
		argv[0] = "wlanctl";
		argv[1] = cfg.device;
		argv[2] = "lnxreq_wlansniff";
		argv[3] = "enable=false";
		argv[4] = NULL;
		argc = 4;
		wlanctl (argc, argv);
		memset (argv, 0, 10);
		argv[0] = "wlanctl";
		argv[1] = cfg.device;
		argv[2] = "lnxreq_ifstate";
		argv[3] = "ifstate=disable";
		argv[4] = NULL;
		wlanctl (argc, argv);
	break;
#endif
	case DT_CISCO:
	{
		FILE *cisco_config;
		snprintf(syscmd, 120, "/proc/driver/aironet/%s/Config", cfg.device);
		if ((cisco_config = fopen(syscmd, "w")) == NULL) {
			return FALSE;
		}
		fprintf(cisco_config, "XmitPower: 100\n");
		fclose(cisco_config);
	}
	break;
	}			// switch
	RawSock = -1;
#ifdef DEBUG
	fprintf (stderr,
				 "Finshed shutting down card.\n");
#endif	
}


int
selectChannel (int channel)
{
	int fd = -1;
	struct iwreq ireq;	//for Orinoco
	int *ptr;
	char syscmd[30];
#ifdef WLAN_NG
	char Channel[25];
	char *argv[10];
	int argc;
#endif

	/* don't change channel if not necessary */
	if (channel == lastchan)
		return 0;
	
	lastchan = channel;
	
	switch (cfg.devtype)
	{
	case DT_HOSTAP:
		if (channel < 14)
		{
			sprintf (syscmd, "/sbin/iwconfig %s channel %i", cfg.device, channel);
			system (syscmd);
		}
		break;

	case DT_ORINOCO:
		fd = socket (AF_INET, SOCK_STREAM, 0);
		if (fd == -1)
		{
			if (!QUIET_MODE) fprintf (stderr,
				 "Error opening socket for orinoco card: %s.\n",
				 strerror (errno));
			return -1;
		}
		ptr = (int *) ireq.u.name;
		ptr[0] = 1;
		ptr[1] = channel;
		strcpy (ireq.ifr_ifrn.ifrn_name, cfg.device);
		if (ioctl (fd, SIOCIWFIRSTPRIV + 0x8, &ireq) != 0)
		{
			if (!QUIET_MODE) fprintf (stderr,
				 "Error setting channel for orinoco card: %s.\n",
				 strerror (errno));
		}
		close (fd);
		break;
#ifdef WLAN_NG
	case DT_PRISM:
		memset (argv, 0, 10);
		sprintf (Channel, "channel=%d", channel);

		argv[0] = "wlanctl";
		argv[1] = cfg.device;
		argv[2] = "lnxreq_wlansniff";
		argv[3] = "enable=false";
		argv[4] = NULL;
		argc = 4;
		wlanctl (argc, argv);

		memset (argv, 0, 10);
		argv[0] = "wlanctl";
		argv[1] = cfg.device;
		argv[2] = "lnxreq_wlansniff";
		argv[3] = Channel;
		argv[4] = "enable=true";
		argv[5] = "prismheader=true";
		argv[6] = "keepwepflags=false";
		argv[7] = NULL;
		argc = 7;
		wlanctl (argc, argv);
		break;
#endif
	}
	return 0;
}

static int
openCard (void)
{
	char syscmd[120];
	int fd = -1;
	struct iwreq ireq;	//for Orinoco
	int *ptr;

#ifdef DEBUG
	fprintf (stderr, "Opening device %s\n", cfg.device);
#endif
	sprintf (syscmd, "/sbin/ifconfig %s up", cfg.device);
	system (syscmd);
	usleep(10000);	
	
	switch (cfg.devtype)
	{
	case DT_SCAN:
		sprintf (syscmd, "/sbin/iwconfig %s mode managed", cfg.device);
		system (syscmd);
		sleep(1);	
		sprintf (syscmd, "/sbin/iwconfig %s essid any", cfg.device);
		system (syscmd);
		return (TRUE);
	break;
	
	case DT_HOSTAP:
		sprintf (syscmd, "/sbin/iwpriv %s monitor 3", cfg.device);
		system (syscmd);
		sleep(1);	
		sprintf (syscmd, "/sbin/iwconfig %s mode monitor", cfg.device);
		system (syscmd);
		return (TRUE);
	break;

	case DT_ORINOCO:
		fd = socket (AF_INET, SOCK_STREAM, 0);
		if (fd == -1)
		{
			if (!QUIET_MODE) fprintf (stderr,
				 "Error opening socket for orinoco card: %s.\n",
				 strerror (errno));
			return FALSE;
		}
		ptr = (int *) ireq.u.name;
		ptr[0] = 1;
		ptr[1] = 1;
		strcpy (ireq.ifr_ifrn.ifrn_name, cfg.device);
		if (ioctl (fd, SIOCIWFIRSTPRIV + 0x8, &ireq) != 0)
		{
			if (!QUIET_MODE) fprintf (stderr,
				 "Error setting channel for orinoco card: %s.\n",
				 strerror (errno));
			return FALSE;
		}
		close (fd);
		return TRUE;
		break;

#ifdef WLAN_NG
	case DT_PRISM:
	{
		char *argv[10];
		int argc;
		memset (argv, 0, 10);
		sprintf (syscmd, "/sbin/ifconfig %s up", cfg.device);
		argv[0] = "wlanctl";
		argv[1] = cfg.device;
		argv[2] = "lnxreq_ifstate";
		argv[3] = "ifstate=enable";
		argv[4] = NULL;
		argc = 4;
		wlanctl (argc, argv);
		fd = system (syscmd);
		return (fd==0);
		break;
	}
#endif
	case DT_CISCO: /* from kismet */
		{
			FILE *cisco_config;
			snprintf(syscmd, 120, "/proc/driver/aironet/%s/Config", cfg.device);
	
			if ((cisco_config = fopen(syscmd, "w")) == NULL) {
				return FALSE;
			}
			fprintf(cisco_config, "Mode: r\n");
			fprintf(cisco_config, "Mode: y\n");
			fprintf(cisco_config, "XmitPower: 1\n");
			fclose(cisco_config);
			return TRUE;
			break;
		}
	}
	return FALSE;
}
