/*
 * prismstumbler wireless lan analyzing tool
 *
 * Copyright (C) 2000 and created by Jan Fernquist  <Jan.B.Fernquist@telia.com>
 * Copyright (C) 2003 Florian Boor <florian.boor@kernelconcepts.de>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 * Wireless extensions scan interface.
 *
 * Most stuff is taken from wireless tools,
 * Copyright (c) 1997-2002 Jean Tourrilhes <jt@hpl.hp.com>  
 * 
 */
 
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <sys/time.h>
#include <sys/types.h>
#include "iface-scan.h"
#include "iwlib.h"
#include "prismstumbler.h"

static ScanResult_t *results;
static int num_nets = 0;
extern psconfig_t cfg;
extern void send_netlist (int sock, int all); /* prismstumbler.c */


/*
 * Print one element from the scanning results
 */
static inline int
print_scanning_token(struct iw_event *	event,	/* Extracted token */
		     int		ap_num,	/* AP number */
		     struct iw_range *	iwrange,	/* Range info */
		     int		has_range)
{
  char		buffer[128];	/* Temporary buffer */

  /* Now, let's decode the event */
  switch(event->cmd)
    {
    case SIOCGIWAP:
#ifdef DEBUG		
      printf("          Cell %02d - Address: %s\n", ap_num,
	     iw_pr_ether(buffer, event->u.ap_addr.sa_data));
#endif	
      ap_num++;
      num_nets++;
      results = realloc(results,num_nets * sizeof(ScanResult_t));
      memset(&results[num_nets-1],0,sizeof(ScanResult_t));
      snprintf(results[num_nets-1].BssId,20,
		iw_pr_ether(buffer, event->u.ap_addr.sa_data)); 
      time(&results[num_nets-1].when);
      break;
#ifdef DEBUG		
    case SIOCGIWNWID:
      if(event->u.nwid.disabled)
	printf("                    NWID:off/any\n");
      else
	printf("                    NWID:%X\n", event->u.nwid.value);
      break;
#endif	  
    case SIOCGIWFREQ:
      {
	double		freq;			/* Frequency/channel */
	freq = iw_freq2float(&(event->u.freq));
	if ((freq > 15.0) || (freq < 1))
		results[num_nets-1].Channel = iw_freq_to_channel(freq,iwrange);
	else
		results[num_nets-1].Channel = (u_char)freq;
#ifdef DEBUG		
	iw_print_freq(buffer, freq);
	printf("                    %s\n", buffer);
#endif		  
      }
      break;
    case SIOCGIWMODE:
#ifdef DEBUG		
      printf("                    Mode:%s\n",
	     iw_operation_mode[event->u.mode]);
#endif	
		 switch (event->u.mode)
		 {
			 case 1: /* ad-hoc */
			 	results[num_nets-1].isAdHoc = 1;
			 break;
			 case 3: /* master */
			 	results[num_nets-1].isAp = 1;
			 break;
			 case 4:
			 case 5: /* secondary, repeater */
			 	results[num_nets-1].isAp = 2;
			 break;
		 } 
      break;
#ifdef DEBUG		
    case SIOCGIWNAME:
      printf("                    Protocol:%-1.16s\n", event->u.name);
      break;
#endif	
    case SIOCGIWESSID:
      {
	char essid[IW_ESSID_MAX_SIZE+1];
	if((event->u.essid.pointer) && (event->u.essid.length))
	  memcpy(essid, event->u.essid.pointer, event->u.essid.length);
	essid[event->u.essid.length] = '\0';
	if(event->u.essid.flags)
	  {
	    /* Does it have an ESSID index ? */
#ifdef DEBUG		
	    if((event->u.essid.flags & IW_ENCODE_INDEX) > 1)
	      printf("                    ESSID:\"%s\" [%d]\n", essid,
		     (event->u.essid.flags & IW_ENCODE_INDEX));
	    else
	      printf("                    ESSID:\"%s\"\n", essid);
#endif		
	      snprintf(results[num_nets-1].SSID,32,essid); 	
	  }
	else
	{
#ifdef DEBUG		
	  printf("                    ESSID:off/any\n");
#endif		
      snprintf(results[num_nets-1].SSID,32,"<off>"); 	
	}
      }
      break;
    case SIOCGIWENCODE:
      {
	unsigned char	key[IW_ENCODING_TOKEN_MAX];
	if(event->u.data.pointer)
	  memcpy(key, event->u.essid.pointer, event->u.data.length);
	else
	  event->u.data.flags |= IW_ENCODE_NOKEY;
#ifdef DEBUG		
	printf("                    Encryption key:");
	if(event->u.data.flags & IW_ENCODE_DISABLED)
	  printf("off\n");
	else
	  {
	    /* Display the key */
	    iw_print_key(buffer, key, event->u.data.length,
			 event->u.data.flags);
	    printf("%s", buffer);
	    /* Other info... */
	    if((event->u.data.flags & IW_ENCODE_INDEX) > 1)
	      printf(" [%d]", event->u.data.flags & IW_ENCODE_INDEX);
	    if(event->u.data.flags & IW_ENCODE_RESTRICTED)
	      printf("   Security mode:restricted");
	    if(event->u.data.flags & IW_ENCODE_OPEN)
	      printf("   Security mode:open");
	    printf("\n");
		results[num_nets-1].hasWep = 1;
	  }
#endif	  
		if(!(event->u.data.flags & IW_ENCODE_DISABLED))
			results[num_nets-1].hasWep = 1;
      }
      break;
    case SIOCGIWRATE:
#ifdef DEBUG		
      iw_print_bitrate(buffer, event->u.bitrate.value);
      printf("                    Bit Rate:%s\n", buffer);
#endif	
	  if (event->u.bitrate.value/KILO > results[num_nets-1].speed)
		  results[num_nets-1].speed = event->u.bitrate.value/KILO;
      break;
    case IWEVQUAL:
      {
//	event->u.qual.updated = 0x0;	/* Not that reliable, disable */
		  results[num_nets-1].Signal = event->u.qual.updated;
#ifdef DEBUG		
	iw_print_stats(buffer, &event->u.qual, iwrange, has_range);
	printf("                    %s\n", buffer);
#endif		  
	break;
      }
#ifdef DEBUG		
#if WIRELESS_EXT > 14
    case IWEVCUSTOM:
      {
	char custom[IW_CUSTOM_MAX+1];
	if((event->u.data.pointer) && (event->u.data.length))
	  memcpy(custom, event->u.data.pointer, event->u.data.length);
	custom[event->u.data.length] = '\0';
	printf("                    Extra:%s\n", custom);
      }
      break;
#endif /* WIRELESS_EXT > 14 */
#endif
	  default:
      printf("                    (Unknown Wireless Token 0x%04X)\n",
	     event->cmd);
   }	/* switch(event->cmd) */

  /* May have changed */
  return(ap_num);
}


/*
 * Perform a scanning on one device
 */
static int
print_scanning_info(int		skfd,
		    char *	ifname,
		    char *	args[],		/* Command line args */
		    int		count)		/* Args count */
{
  struct iwreq		wrq;
  unsigned char		buffer[IW_SCAN_MAX_DATA];	/* Results */
  struct timeval	tv;				/* Select timeout */
  int			timeout = 5000000;		/* 5s */

  /* Avoid "Unused parameter" warning */
  args = args; count = count;

  /* Init timeout value -> 250ms*/
  tv.tv_sec = 0;
  tv.tv_usec = 250000;

  /*
   * Here we should look at the command line args and set the IW_SCAN_ flags
   * properly
   */
  wrq.u.param.flags = IW_SCAN_DEFAULT;
  wrq.u.param.value = 0;		/* Later */

  /* Initiate Scanning */
  if(iw_set_ext(skfd, ifname, SIOCSIWSCAN, &wrq) < 0)
    {
      if(errno != EPERM)
	{
	  fprintf(stderr, "%-8.8s  Interface doesn't support scanning : %s\n\n",
		  ifname, strerror(errno));
	  return(-1);
	}
      /* If we don't have the permission to initiate the scan, we may
       * still have permission to read left-over results.
       * But, don't wait !!! */
      tv.tv_usec = 0;
    }
  timeout -= tv.tv_usec;

  /* Forever */
  while(1)
    {
      fd_set		rfds;		/* File descriptors for select */
      int		last_fd;	/* Last fd */
      int		ret;

      /* Guess what ? We must re-generate rfds each time */
      FD_ZERO(&rfds);
      last_fd = -1;

      /* In here, add the rtnetlink fd in the list */

      /* Wait until something happens */
      ret = select(last_fd + 1, &rfds, NULL, NULL, &tv);

      /* Check if there was an error */
      if(ret < 0)
	{
	  if(errno == EAGAIN || errno == EINTR)
	    continue;
	  fprintf(stderr, "Unhandled signal - exiting...\n");
	  return(-1);
	}

      /* Check if there was a timeout */
      if(ret == 0)
	{
	  /* Try to read the results */
	  wrq.u.data.pointer = buffer;
	  wrq.u.data.flags = 0;
	  wrq.u.data.length = sizeof(buffer);
	  if(iw_get_ext(skfd, ifname, SIOCGIWSCAN, &wrq) < 0)
	    {
	      /* Check if results not available yet */
	      if(errno == EAGAIN)
		{
		  /* Restart timer for only 100ms*/
		  tv.tv_sec = 0;
		  tv.tv_usec = 100000;
		  timeout -= tv.tv_usec;
		  if(timeout > 0)
		    continue;	/* Try again later */
		}

	      /* Bad error */
	      fprintf(stderr, "%-8.8s  Failed to read scan data : %s\n\n",
		      ifname, strerror(errno));
	      return(-2);
	    }
	  else
	    /* We have the results, go to process them */
	    break;
	}

      /* In here, check if event and event type
       * if scan event, read results. All errors bad & no reset timeout */
    }

  if(wrq.u.data.length)
    {
      struct iw_event		iwe;
      struct stream_descr	stream;
      int			ap_num = 1;
      int			ret;
      struct iw_range		range;
      int			has_range;
      has_range = (iw_get_range_info(skfd, ifname, &range) >= 0);
#ifdef DEBUG		
      printf("%-8.8s  Scan completed :\n", ifname);
#endif	  
      iw_init_event_stream(&stream, buffer, wrq.u.data.length);
      do
	{
	  /* Extract an event and print it */
	  ret = iw_extract_event_stream(&stream, &iwe);
	  if(ret > 0)
	    ap_num = print_scanning_token(&iwe, ap_num, &range, has_range);
	}
      while(ret > 0);
#ifdef DEBUG				  
      printf("\n");
#endif	  
    }
#ifdef DEBUG		
  else
    printf("%-8.8s  No scan results\n", ifname);
#endif
  return(0);
}


/*
	Perform scan on given device.
	Returns TRUE on sucess, FALSE if scanning is not available.
*/
int 
do_scan_interface(char *ifname, int ps_socket, float longitude, float latitude)
{
	int i, result;
	int cardfd;
	
	/* Create a channel to the NET kernel. */
	if((cardfd = iw_sockets_open()) < 0)
	{
		perror("socket");
		return 0;
	}
	
	result = print_scanning_info(cardfd,ifname, NULL, 0);
	for (i = 0;i < num_nets; i++)
	{
		results[i].longitude = longitude;
		results[i].latitude = latitude;
		update_all (&results[i], ps_socket);
		send_netlist (ps_socket, 0);
	}
	
	free(results);
	results = NULL;
	num_nets = 0;
	close(cardfd);
	return (result == 0);
}
